// A big thank you to Craig Andera for this one!
// http://staff.develop.com/candera/weblog/stories/2003/02/20/theLastConfigurationSectionHandlerIllEverNeed.html

namespace Madgeek.Configuration
{
    #region Imports
    using System;
    using System.Configuration;
    using System.Xml;
    using System.Xml.Serialization;
    using System.Xml.XPath;
    #endregion

    /// <summary>
    /// Configuration section handler that deserializes connfiguration settings to an object.
    /// </summary>
    /// <remarks>The configuration node must have a type attribute defining the type to deserialize to.</remarks>
    public class XmlSerializerSectionHandler : IConfigurationSectionHandler
    {
        /// <summary>
        /// Implemented by all configuration section handlers to parse the XML of the configuration section.
        /// The returned object is added to the configuration collection and is accessed by <see cref="System.Configuration.ConfigurationSettings.GetConfig"/>.
        /// </summary>
        /// <param name="parent">The configuration settings in a corresponding parent configuration section.</param>
        /// <param name="configContext">An <see cref="System.Web.Configuration.HttpConfigurationContext"/> when this method is called from the ASP.NET configuration system. Otherwise, this parameter is reserved and is a null reference (Nothing in Visual Basic).</param>
        /// <param name="section">The <see cref="System.Xml.XmlNode"/> that contains the configuration information from the configuration file. Provides direct access to the XML contents of the configuration section.</param>
        /// <returns>A configuration object.</returns>
        public object Create(object parent, object configContext, System.Xml.XmlNode section)
        {
            XPathNavigator navigator;
            XmlSerializer serializer;
            Type type;
            string typeName;

            navigator = section.CreateNavigator();
            typeName = (string)navigator.Evaluate("string(@type)");
            type = Type.GetType(typeName, true);
            serializer = new XmlSerializer(type);
            return serializer.Deserialize(new XmlNodeReader(section));
        }
    }
}

// Author: Fabrice Marguerie
// http://weblogs.asp.net/fmarguerie/
// fabrice@madgeek.com
//
// Free for use
//
// Based on code from Fritz Onion: http://pluralsight.com/blogs/fritz/archive/2004/07/21/1651.aspx

namespace Madgeek.Web
{

    #region Imports
    using System;
    using System.Collections;
    using System.Configuration;
    using System.Reflection;
    using System.Text.RegularExpressions;
    using System.Web;
    using System.Xml.Serialization;
    #endregion


	public class Redirection
	{
		[XmlAttribute("targetUrl")]
		public string targetUrl;

		[XmlAttribute("destinationUrl")]
		public string destinationUrl;
		
		[XmlAttribute("permanent")]
		public bool permanent = false;

		[XmlAttribute("ignoreCase")]
		public bool ignoreCase = false;

		[XmlIgnore]
		public Regex	Regex;
	}

	[XmlRoot("redirections")]
	public class ConfigRedirections
	{
		[XmlElement("add")]
		public Redirection[] Redirections;
	}

	/// <summary>
	/// The purpose of this module is to redirect requests 
	/// based on web.config data in the &lt;redirections&gt; section.
	/// </summary>
  public class RedirectModule : IHttpModule
  {
		private string		_ApplicationPath;
		private ArrayList	_Redirections;

		public RedirectModule()
		{
			_ApplicationPath = (HttpRuntime.AppDomainAppVirtualPath.Length > 1) ? HttpRuntime.AppDomainAppVirtualPath : String.Empty;
		}

		#region Private members

		private string HandleTilde(string url)
		{
			if ((url == null) || (url.Length < 1))
				return url;

			if (url.StartsWith("^~/"))
				return "^"+_ApplicationPath+url.Substring("^~/".Length-1);
			else if (url.StartsWith("~/"))
				return _ApplicationPath+url.Substring("~/".Length-1);
			return url;
		}

		private void OnBeginRequest(object sender, EventArgs e)
		{
			HttpApplication	app;
			string					requestUrl;

			app = sender as HttpApplication;
			requestUrl = app.Request.RawUrl;

			foreach (Redirection redirection in _Redirections)
			{
				if (redirection.Regex.IsMatch(requestUrl))
				{
					string	destinationUrl;

					destinationUrl = redirection.Regex.Replace(requestUrl, redirection.destinationUrl, 1);
					destinationUrl = HandleTilde(destinationUrl);

					if (redirection.permanent)
					{
						app.Response.StatusCode = 301; // make a permanent redirect
						app.Response.AddHeader("Location", destinationUrl);
						app.Response.End();
					}
					else
					{
						app.Context.RewritePath(destinationUrl);

						// Keep track of the virtual URL because we'll need it to fix postbacks
						// See http://weblogs.asp.net/jezell/archive/2004/03/15/90045.aspx
						app.Context.Items["VirtualUrl"] = requestUrl;
					}

					break;
				}
			}
		}

		#endregion

		#region IHttpModule members

		public void Init(System.Web.HttpApplication context)
		{
			ConfigRedirections	redirections;

			_Redirections = new ArrayList();

			redirections = (ConfigRedirections) ConfigurationSettings.GetConfig("redirections");
			foreach (Redirection redirection in redirections.Redirections)
			{
				string	targetUrl;

				targetUrl = redirection.targetUrl;
				targetUrl = HandleTilde(targetUrl);
				if (redirection.ignoreCase)
					redirection.Regex = new Regex(targetUrl, RegexOptions.IgnoreCase /* | RegexOptions.Compiled*/);
				else
					redirection.Regex = new Regex(targetUrl/*, RegexOptions.Compiled*/);
				_Redirections.Add(redirection);
			}

			context.BeginRequest += new EventHandler(OnBeginRequest);
		}

		public void Dispose()
		{
		}

		#endregion
	}
}